-- Safe translation helper
local function _t(self, key, fallback)
    local i18n = (self ~= nil and self.i18n) or nil
    if i18n == nil then i18n = g_i18n end
    if i18n ~= nil and i18n.getText ~= nil then
        local ok, res = pcall(i18n.getText, i18n, key)
        if ok and res ~= nil then return res end
    end
    return fallback or key
end

-- FS25 Invoices - Fixed Final Version
InGameMenuInvoices = {}
local InGameMenuInvoices_mt = Class(InGameMenuInvoices, TabbedMenuFrameElement)

local function getByName(cell, name)
    if cell == nil then return nil end
    if cell.getDescendantByName ~= nil then
        return cell:getDescendantByName(name)
    end
    return nil
end

local function blink(text)
    if g_currentMission ~= nil and g_currentMission.showBlinkingWarning ~= nil then
        g_currentMission:showBlinkingWarning(text, 3000)
    end
end

function InGameMenuInvoices.new(i18n, messageCenter)
    local self = InGameMenuInvoices:superClass().new(nil, InGameMenuInvoices_mt)
    self.i18n = i18n or g_i18n
    self.messageCenter = messageCenter
    self.items = {}
    self.openInvoices = {}
    self.paidInvoices = {}
    self.selectedInvoice = nil
    self.createMode = false

    -- Button Definitionen
    self.backButtonInfo = { inputAction = InputAction.MENU_BACK }
    self.payButton = { inputAction = InputAction.MENU_ACTIVATE, text = _t(self, "ui_inv_pay", "Bezahlen"), callback = function() self:onPay() end }
    self.cancelButton = { inputAction = InputAction.MENU_EXTRA_1, text = _t(self, "ui_inv_cancel", "Stornieren"), callback = function() self:onCancel() end }
    self.createButton = { inputAction = InputAction.MENU_EXTRA_2, text = _t(self, "ui_inv_create", "Erstellen"), callback = function() self:onToggleCreate() end }

    self.menuButtons = { self.backButtonInfo, self.payButton, self.cancelButton, self.createButton }
    self:setMenuButtonInfo(self.menuButtons)
    return self
end

function InGameMenuInvoices:onGuiSetupFinished()
    InGameMenuInvoices:superClass().onGuiSetupFinished(self)
    
    -- Sicherer Zugriff auf die Elemente (Behebt LUA-Error Zeile 51)
    self.invoiceTable = self:getDescendantByName("invoiceTable")
    self.createPanel = self:getDescendantByName("createPanel")
    
    local target = self.createPanel or self
    self.receiverInput = target:getDescendantByName("receiverInput")
    self.amountInput = target:getDescendantByName("amountInput")
    self.dueInput = target:getDescendantByName("dueInput")
    self.textInput = target:getDescendantByName("textInput")
end

function InGameMenuInvoices:onFrameOpen()
    InGameMenuInvoices:superClass().onFrameOpen(self)
    if self.invoiceTable ~= nil then
        self.invoiceTable:setDataSource(self)
        self.invoiceTable:setDelegate(self)
    end
    self:refresh()
end

function InGameMenuInvoices:refresh()
    if g_invoices ~= nil and g_invoices.getAllSorted ~= nil then
        self.items = g_invoices:getAllSorted()
    else
        self.items = {}
    end
    self.openInvoices = {}
    self.paidInvoices = {}
    for _, inv in ipairs(self.items) do
        local isPaid = (inv.state == 1)
        if isPaid then table.insert(self.paidInvoices, inv) else table.insert(self.openInvoices, inv) end
    end
    if self.invoiceTable ~= nil then self.invoiceTable:reloadData() end
    self:updateButtons()
end

function InGameMenuInvoices:updateButtons()
    local hasSel = (self.selectedInvoice ~= nil)
    local isPaid = self.selectedInvoice and (self.selectedInvoice.state == 1)
    self.payButton.disabled = not hasSel or isPaid
    self.cancelButton.disabled = not hasSel or isPaid
    
    if self.createMode then
        self.createButton.text = _t(self, "ui_inv_submit", "Absenden")
    else
        self.createButton.text = _t(self, "ui_inv_create", "Erstellen")
    end
    self:setMenuButtonInfo(self.menuButtons)
end

function InGameMenuInvoices:onToggleCreate()
    if self.createPanel == nil then return end
    if not self.createMode then
        self.createMode = true
        self.createPanel:setVisible(true)
    else
        self:submitCreate()
    end
    self:updateButtons()
end

function InGameMenuInvoices:submitCreate()
    if self.receiverInput == nil or self.amountInput == nil then return end
    local receiver = self.receiverInput:getText()
    local amount = tonumber(self.amountInput:getText())
    local dueDays = tonumber(self.dueInput:getText()) or 7
    local text = self.textInput:getText()
    
    if receiver ~= "" and amount ~= nil and amount > 0 then
        if CreateInvoiceEvent ~= nil then
            g_client:getServerConnection():sendEvent(CreateInvoiceEvent.new(receiver, amount, dueDays, text))
        end
        self.receiverInput:setText("")
        self.amountInput:setText("")
        self.dueInput:setText("7")
        self.textInput:setText("")
        self.createMode = false
        if self.createPanel then self.createPanel:setVisible(false) end
        self:refresh()
    else
        blink("Bitte Empfänger und gültigen Betrag eingeben!")
    end
end

function InGameMenuInvoices:onPay()
    if self.selectedInvoice and PayInvoiceEvent ~= nil then
        g_client:getServerConnection():sendEvent(PayInvoiceEvent.new(self.selectedInvoice.id))
        self:refresh()
    end
end

function InGameMenuInvoices:onCancel()
    if self.selectedInvoice and CancelInvoiceEvent ~= nil then
        g_client:getServerConnection():sendEvent(CancelInvoiceEvent.new(self.selectedInvoice.id))
        self:refresh()
    end
end

function InGameMenuInvoices:getNumberOfSections() return 2 end
function InGameMenuInvoices:getNumberOfItemsInSection(list, section)
    if section == 1 then return #self.openInvoices end
    return #self.paidInvoices
end
function InGameMenuInvoices:getTitleForSectionHeader(list, section)
    return (section == 1) and _t(self, "ui_inv_section_open", "Offen") or _t(self, "ui_inv_section_paid", "Bezahlt")
end
function InGameMenuInvoices:getCellTypeForItemInSection(list, section, index) return "rowTemplate" end
function InGameMenuInvoices:getCellTypeForSectionHeader(list, section) return "section" end

function InGameMenuInvoices:populateCellForItemInSection(list, section, index, cell)
    local inv = (section == 1) and self.openInvoices[index] or self.paidInvoices[index]
    if inv ~= nil then
        local idEl = getByName(cell, "id")
        local fromEl = getByName(cell, "from")
        local toEl = getByName(cell, "to")
        local amountEl = getByName(cell, "amount")
        local statusEl = getByName(cell, "status")
        
        if idEl then idEl:setText(tostring(inv.id or "")) end
        if fromEl then fromEl:setText(tostring(inv.from or "Farm " .. (inv.senderFarmId or ""))) end
        if toEl then toEl:setText(tostring(inv.to or "Farm " .. (inv.receiverFarmId or ""))) end
        if amountEl then amountEl:setText(g_i18n:formatMoney(inv.amount or 0)) end
        if statusEl then statusEl:setText(section == 2 and "BEZAHLT" or (inv.stateText or "OFFEN")) end
    end
end

function InGameMenuInvoices:onListSelectionChanged(list, section, index)
    self.selectedInvoice = (section == 1) and self.openInvoices[index] or self.paidInvoices[index]
    self:updateButtons()
end