Invoices = {}
Invoices_mt = Class(Invoices)

Invoices.STATE_OPEN = 0
Invoices.STATE_PAID = 1
Invoices.STATE_OVERDUE = 2

function Invoices.new()
    local self = setmetatable({}, Invoices_mt)
    self.items = {}
    self.nextId = 100000
    return self
end

function Invoices:_allocId()
    self.nextId = self.nextId + 1
    return self.nextId
end

function Invoices:add(inv)
    table.insert(self.items, inv)
    self.nextId = math.max(self.nextId, inv.id or 0)
end

function Invoices:create(senderFarmId, receiverFarmId, amount, text, dueDays)
    local envDay = (g_currentMission ~= nil and g_currentMission.environment ~= nil) and g_currentMission.environment.currentDay or 1
    local id = self:_allocId()
    local inv = {
        id = id,
        senderFarmId = senderFarmId,
        receiverFarmId = receiverFarmId,
        from = string.format("Farm %d", senderFarmId),
        to = string.format("Farm %d", receiverFarmId),
        amount = amount,
        due = envDay + (dueDays or 3),
        state = Invoices.STATE_OPEN,
        text = text or ""
    }
    self:add(inv)
    return inv
end

function Invoices:getById(id)
    for _,inv in ipairs(self.items) do
        if inv.id == id then return inv end
    end
    return nil
end

function Invoices:removeById(id)
    for i=#self.items,1,-1 do
        if self.items[i].id == id then
            table.remove(self.items, i)
            return true
        end
    end
    return false
end

function Invoices:_updateOverdue(inv)
    if inv.state == Invoices.STATE_PAID then return end
    local envDay = (g_currentMission ~= nil and g_currentMission.environment ~= nil) and g_currentMission.environment.currentDay or 1
    if envDay > (inv.due or 0) then
        inv.state = Invoices.STATE_OVERDUE
    else
        inv.state = Invoices.STATE_OPEN
    end
end

function Invoices:updateAllOverdue()
    for _,inv in ipairs(self.items) do
        self:_updateOverdue(inv)
    end
end

function Invoices:getOpenOrOverdue()
    local t = {}
    for _,inv in ipairs(self.items) do
        self:_updateOverdue(inv)
        if inv.state ~= Invoices.STATE_PAID then
            table.insert(t, inv)
        end
    end
    return t
end

function Invoices:getPaid()
    local t = {}
    for _,inv in ipairs(self.items) do
        if inv.state == Invoices.STATE_PAID then table.insert(t, inv) end
    end
    return t
end

-- server-side money transfer
function Invoices:payServer(id)
    local inv = self:getById(id)
    if inv == nil or inv.state == Invoices.STATE_PAID then
        return false, "invalid"
    end

    local payer = g_farmManager ~= nil and g_farmManager:getFarmById(inv.receiverFarmId) or nil
    local payee = g_farmManager ~= nil and g_farmManager:getFarmById(inv.senderFarmId) or nil
    if payer == nil or payee == nil then
        return false, "farms"
    end

    -- optional: check funds (if farm supports getBalance)
    if payer.getBalance ~= nil and payer:getBalance() < inv.amount then
        return false, "funds"
    end

    payer:addMoney(-inv.amount, MoneyType.OTHER)
    payee:addMoney(inv.amount, MoneyType.OTHER)

    inv.state = Invoices.STATE_PAID
    return true
end

-- Savegame
function Invoices:saveToXML(xmlFile, key)
    setXMLInt(xmlFile, key.."#nextId", self.nextId)
    for i,inv in ipairs(self.items) do
        local k = string.format("%s.invoice(%d)", key, i-1)
        setXMLInt(xmlFile, k.."#id", inv.id or 0)
        setXMLInt(xmlFile, k.."#senderFarmId", inv.senderFarmId or 0)
        setXMLInt(xmlFile, k.."#receiverFarmId", inv.receiverFarmId or 0)
        setXMLString(xmlFile, k.."#from", inv.from or "")
        setXMLString(xmlFile, k.."#to", inv.to or "")
        setXMLFloat(xmlFile, k.."#amount", inv.amount or 0)
        setXMLInt(xmlFile, k.."#due", inv.due or 0)
        setXMLInt(xmlFile, k.."#state", inv.state or 0)
        setXMLString(xmlFile, k.."#text", inv.text or "")
    end
end

function Invoices:loadFromXML(xmlFile, key)
    self.items = {}
    self.nextId = getXMLInt(xmlFile, key.."#nextId") or 100000

    local i = 0
    while true do
        local k = string.format("%s.invoice(%d)", key, i)
        if not hasXMLProperty(xmlFile, k) then break end
        local inv = {
            id = getXMLInt(xmlFile, k.."#id") or 0,
            senderFarmId = getXMLInt(xmlFile, k.."#senderFarmId") or 0,
            receiverFarmId = getXMLInt(xmlFile, k.."#receiverFarmId") or 0,
            from = getXMLString(xmlFile, k.."#from") or "",
            to = getXMLString(xmlFile, k.."#to") or "",
            amount = getXMLFloat(xmlFile, k.."#amount") or 0,
            due = getXMLInt(xmlFile, k.."#due") or 0,
            state = getXMLInt(xmlFile, k.."#state") or 0,
            text = getXMLString(xmlFile, k.."#text") or ""
        }
        table.insert(self.items, inv)
        self.nextId = math.max(self.nextId, inv.id or 0)
        i = i + 1
    end
end

g_invoices = Invoices.new()
